from dotenv import load_dotenv
import logging
import os
import sys

import requests
from playwright.sync_api import sync_playwright
import time

log = None
development_mode = os.getenv('PYCHARM_HOSTED')
load_dotenv()

login_user = os.environ['LOGIN_USER']
login_pass = os.environ['LOGIN_PASS']

def snatch_grading_work(headless=None):
    global log

    if log is None:
        log = logging.getLogger('snatcher')

    if headless is None:
        headless = False if development_mode else True
    log.info(f"Snatching grading work ({headless})")

    playwright = sync_playwright().start()
    browser = playwright.chromium.launch(headless=headless)
    page = browser.new_page()

    try:
        log.info("Going over to SDO")
        page.goto('https://eco.scrum.org/grading/results', wait_until='networkidle', timeout=30000)

        # Check for password input and perform login if present
        password_input = page.query_selector('input[id="password"]')
        if password_input:
            log.info("Logging in")
            page.fill('input[id="password"]', login_pass)
            page.fill('input[id="loginId"]', login_user)
            page.click('button:has-text("Log In")')

        # Wait for a specific XHR request to complete
        with page.expect_response('https://as-sdo-eco-api-prod-eastus.azurewebsites.net/grading/results', timeout=30000) as response_info:
            response = response_info.value
            log.info(f"XHR request completed with status: {response.status}")

        page.wait_for_selector('a:has-text("Grading Queue")', timeout=15000)
        page.wait_for_selector('span:has-text("Result ID")', timeout=15000)

        time.sleep(2)

        # Don't claim if we already have one claimed
        unclaim_buttons = page.query_selector_all('span:text-is("Release My Claim")')
        unclaim_count = len(unclaim_buttons)
        log.info(f"Release My Claim buttons: {unclaim_count}")
        if unclaim_count > 0:
            log.info("Not claiming yet another test")

        else:
            log.info("No grading jobs assigned to me, going ahead to snatch a new one")

            # Verify the number of "Claim" buttons
            claim_buttons = page.query_selector_all('span:text-is("Claim")')
            claim_count = len(claim_buttons)
            log.info(f"Claim buttons: {claim_count}")
            if claim_count == 0:
                log.info("No Claim button(s) found - nothing to do")
            else:
                log.info("Clicking Claim button 0")
                claim_buttons[0].click()
                # Wait for page to reload and number of "Claim" buttons to reduce by 1
                start_time = time.time()
                while True:
                    current_buttons = page.query_selector_all('span:text-is("Claim")')
                    if len(current_buttons) == claim_count - 1:
                        break
                    if time.time() - start_time > 30:  # Timeout after 30 seconds
                        break
                    time.sleep(0.5)  # Poll every 0.5 seconds
                log.info(f"Clicked first 'Claim' button; 'Claim' buttons reduced to {claim_count - 1}")
                requests.post("https://push.wendt.io", params={"message": "New grading work snatched!"})

    except Exception as e:
        log.error(f"Error: {e}")

    if development_mode:
        input("Press Enter to continue...")

    browser.close()
    playwright.stop()
    log.info("Done snatching grading work")


if __name__ == '__main__':
    logging.basicConfig(level=logging.INFO, stream=sys.stdout)
    snatch_grading_work()
