from dotenv import load_dotenv
import email
import imaplib
import json
import os
import time
import datetime

import requests


class Checker:

    # Function to fetch emails with a specific label
    def __init__(self):
        self.api_46elks_user = os.environ['API_46ELKS_USER']
        self.api_46elks_pass = os.environ['API_46ELKS_PASS']
        self.imap_user = os.environ['IMAP_USER']
        self.imap_pass = os.environ['IMAP_PASS']
        self.mail = None

    def notify_fredrik_via_sms(self, message='Level III Grading Opportunity\n\nGo to https://eco.scrum.org/grading/results'):
        print("new grading opportunity")
        fields = {
            'from': '46702778511',
            'to': '46702778511',
            'message': message
        }
        response = requests.post("https://api.46elks.com/a1/SMS", data=fields,
                                 auth=(self.api_46elks_user, self.api_46elks_pass))

        try:
            if response.json()['status'] == "created":
                print("SMS sent")
        except ValueError:
            print(f"SMS fail: {response.content}")

    def check_for_new_email(self):
        print(f"checking for new email {datetime.datetime.now().isoformat()}")
        if not self.mail:
            self.setup_mail_connection()

        (typ, [data]) = self.mail.select("\"" + "Scrum.org/Level III Grading" + "\"")

        if typ != "OK":
            raise Exception(f"Hmm {typ} was not 'OK'")

        count = int(data)
        print(f"found {count} emails")
        if count == 0:
            return

        # Search for emails in the selected label
        result, data = self.mail.search(None, "ALL")
        string_of_ids = data[0].decode("ascii", "ignore")
        ids = string_of_ids.split()

        ids_seen = set()
        new_ids = set()
        try:
            id_data = open("ids_seen.json").read()
            ids_seen.update(json.loads(id_data))
        except:
            pass

        for email_id in ids:
            result, data = self.mail.fetch(email_id, "(RFC822)")
            raw_email = data[0][1]
            msg = email.message_from_bytes(raw_email)
            msg_id = msg["Message-ID"]
            if msg_id in ids_seen:
                continue
            new_ids.add(msg_id)

        if len(new_ids) > 0:
            self.notify_fredrik_via_sms()
            ids_seen.update(new_ids)
            print(f"Saving {ids_seen}")
            open("ids_seen.json", "w").write(json.dumps(list(ids_seen)))

        self.mail.close()
        self.mail.logout()
        self.mail = None

    def run(self):
        error_count = 0
        while error_count < 3:
            try:
                self.check_for_new_email()
                error_count = 0
            except Exception as e:
                print(f"Error: {e}")
                try:
                    self.mail.close()
                except:
                    pass
                self.mail = None
                error_count += 1
                time.sleep(60 * (error_count + 1))
            time.sleep(60)
        self.notify_fredrik_via_sms('Exiting grading monitoring')
        print("3 errors, aborting")

    def setup_mail_connection(self):
        print("Setting up new connection to gmail.com")
        self.mail = imaplib.IMAP4_SSL('imap.gmail.com')
        self.mail.login(self.imap_user, self.imap_pass)


if __name__ == '__main__':
    load_dotenv()
    print("""How to:
screen
python3 -m venv venv
source venv/bin/activate
pip install requests python-dotenv
cp example.env .env
python3 job.py
""")
    Checker().run()
