#!/usr/bin/env bash

# version 201911270815

set -e


function main {
    init
    prepare
    build
    setup_networkz
    redeploy
    notify_redeploy
}


function init {
    export COMPOSE_PROJECT_NAME=sdo-grading-job

    export COMPOSE_API_VERSION=auto
    export PROJECT=$(basename $(dirname "$(readlink -f "$0")"))

    export BASE_DIR=$(pwd)
    export TARGET_DIR=$BASE_DIR
}


function prepare {
    loggit "Preparing build & deployment environment"

    # make_clean_target_dir
    check_for_upstream_certificates
}


function make_clean_target_dir {
    cd $BASE_DIR
    if [ -d $TARGET_DIR ] ; then rm -rf $TARGET_DIR ; fi
    mkdir -p $TARGET_DIR
}


function build {
    loggit "Building"

    cd $TARGET_DIR
    if [ -f Dockerfile ] ; then
      echo "Building base image $COMPOSE_PROJECT_NAME:latest"
      docker build --no-cache -t $COMPOSE_PROJECT_NAME:latest -f Dockerfile .
    fi
    docker-compose pull
    docker-compose build
}


function setup_networkz {
    loggit "Setting up docker networking"

    # no good way of doing this without exit code != 0
    set +e
    docker network create internetz > /dev/null
    docker network ls
    set -e
}


function redeploy {
    loggit "Deploying"

    cd $TARGET_DIR
    echo
    echo =======================================================================
    echo Running docker containers BEFORE
    echo -----------------------------------------------------------------------
    docker ps
    sleep 2
    echo -----------------------------------------------------------------------

    echo "Starting new containers"
    docker-compose down
    docker-compose up -d

    sleep 5

    echo
    echo =======================================================================
    echo Running docker containers AFTER
    echo -----------------------------------------------------------------------
    docker ps
    sleep 2
    echo -----------------------------------------------------------------------
    echo
}


function notify_redeploy {
    loggit "Sending notification of deploy"

    cd $BASE_DIR
    GIT_MSG=$(git log -1 --pretty=%B)
    MSG="$GIT_MSG $GO_PIPELINE_COUNTER.$GO_STAGE_COUNTER"

    curl -m 5 -fsSL https://push.wendt.io/ \
        -d "message=$PROJECT redeployed: $MSG" \
        -d url=$PUSH_URL || echo "Notification failed"
}


function loggit {
    echo ">>>"
    echo -n ">>> STEP: " 
    echo "$@"
    echo ">>>"
}


function inject_vpn_ip_into_docker_compose_file {
    loggit "Injecting VPN IP into Docker Compose file"

    HOST=$(hostname --short)
    VPNHOST=${HOST}.wendt.vpn
    VPNIP=$(host $VPNHOST | grep address | tail -n 1 | cut -d ' ' -f 4)
    if [ -z "$VPNIP" ] ; then
        echo "Couldn't resolve $VPNHOST to an IP address"
        exit 1
    fi

    if [ "$HOME" = "/home/ceda" ] ; then
        DEBUG=true
    fi
    if [ "$DEBUG" ] ; then
        clear
        echo
        echo "LOCAL DEVELOPMENT DETECTED"
        echo
        VPNIP=0.0.0.0
    fi

    echo -n "Creating build dir: "
    BUILD_DIR=build/${COMPOSE_PROJECT_NAME}
    mkdir -p $BUILD_DIR
    echo "$BUILD_DIR"

    echo -n "Prepping source files: "

    echo -n "${COMPOSE_PROJECT_NAME} config "
    cp -r src/docker/images/ $BUILD_DIR
    mkdir -p $BUILD_DIR/build

    echo -n "docker-compose.yml "
    sed -e "s/__VPNIP__/$VPNIP/" $BASE_DIR/src/docker/runtime/docker-compose.yml.template > $BUILD_DIR/docker-compose.yml

    echo "done"

    tree $BASE_DIR/$BUILD_DIR
    export TARGET_DIR=$BASE_DIR/$BUILD_DIR
}


function check_for_upstream_certificates {

    if [ -d upstream/ ] ; then
        echo "USING CERTIFICATES FROM UPSTREAM JOB (folder upstream)"
        cp -r upstream/certificates/* proxy/certificates/
    else
        echo "USING CERTIFICATES FROM git REPO - old?"
    fi
}


main

